/* Filename: KappaCalculator.java
 * Creator: M.A. Finlayson
 * Format: Java 2 v1.6.0
 * Date created: Feb 8, 2010
 */
package nil.ucm.indications2.ui.agreement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import nil.ucm.indications2.core.rep.IConstituent;
import nil.ucm.indications2.core.rep.IIndicationStructure;
import nil.ucm.indications2.core.rep.INucleus;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.IndicationType;
import nil.ucm.indications2.ui.agreement.AgreeResult.AgreeResultBuilder;
import nil.ucm.indications2.ui.agreement.OverallAgreeResult.OverallAgreeBuilder;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;

import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.parsing.core.rep.token.IToken;
import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.align.Aligner;
import edu.mit.story.core.align.Aligner.Pair;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.IDescSet;
import edu.mit.story.core.desc.IHasSegments;
import edu.mit.story.core.desc.Segment;
import edu.mit.story.core.measure.IResult;
import edu.mit.story.core.measure.PairResult;
import edu.mit.story.core.measure.StoryPairExtractor;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.StoryModelImporter;
import edu.mit.story.core.position.HasPositionSet;
import edu.mit.story.core.position.IHasPositionSet;

/** 
 * TODO: Write comment
 *
 * @author M.A. Finlayson
 * @author R. Hervas
 * @version $Rev$, $LastChangedDate$
 * @since nil.ucm.indications2.ui 1.0.0
 */
public class AgreementCalculator2 {
	
	public List<IResult> calculate(IResource one, IResource two, IProgressMonitor pm){
		
		// pair up files
		List<IResult> results = new LinkedList<IResult>();
		Map<IFile, IFile> pairs = StoryPairExtractor.getInstance().getPairs(one, two, results);
		
		// collect list of folders by which to generate overalls
		Set<IResource> tests = new HashSet<IResource>();
				
		// do counting over files we found
		pm.beginTask("Calculating Agreement", pairs.size()+1);
		String msg;
		for(Entry<IFile, IFile> pair : pairs.entrySet()){
			msg = "Comparing " + pair.getKey().getName() + " with " + pair.getValue().getName();
			tests.add(pair.getKey().getParent());
			System.err.println(msg);
			pm.subTask(msg);
			results.add(calculate(pair.getKey(), pair.getValue()));
			pm.worked(1);
			if(pm.isCanceled()) break;
		}
		
		// dump to list
		List<IResource> testList = new ArrayList<IResource>(tests.size() + 1);
		tests.remove(one);
		testList.addAll(tests);
		testList.add(one);
		
		// calculate final kappas+f-measures
		msg = "Calculating overall agreement";
		pm.subTask(msg);
		for(IResource test : testList) calculateOverall(one, two, results, test);
		pm.worked(1);
		pm.done();

		return results;
		
	}
			
	/**
	 * TODO: Write comment
	 *
	 * @param one
	 * @param two
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public IResult calculate(IFile one, IFile two){
		
		IStoryModel modelOne;
		IStoryModel modelTwo;
		
		List<IResult> results = new LinkedList<IResult>();
		
		modelOne = StoryModelImporter.extractModel(one, results);
		if(modelOne == null){
			IResult error = results.remove(0);
			return new PairResult(one, two, "Unable to get contents of first resource", error.getException());
		}
			
		modelTwo = StoryModelImporter.extractModel(one, results);
		if(modelTwo == null){
			IResult error = results.remove(0);
			return new PairResult(one, two, "Unable to get contents of second resource", error.getException());
		}
		
		try{
			return calculate(one, two, modelOne, modelTwo);
		} catch(Exception e){
			e.printStackTrace();
			return new PairResult(one, two, "Unable to calculate file", e);
		}
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param one
	 * @param two
	 * @param modelOne
	 * @param modelTwo
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public IResult calculate(IFile one, IFile two, IStoryModel modelOne, IStoryModel modelTwo){
		
		AgreeResultBuilder builder = new AgreeResultBuilder();
		
		// pair up tokens
		PairMap<IDesc> tokenPairs = pairTokens(modelOne, modelTwo, builder);
		System.err.println('\t' + Integer.toString(builder.tokensUnpairedOne + builder.tokensUnpairedTwo) + " unpaired tokens");
		
		// pair up references
		PairMap<IReference> refPairs = pairReferences(modelOne, modelTwo, tokenPairs, builder);
		System.err.println('\t' + Integer.toString(builder.refsUnpairedOne + builder.refsUnpairedTwo) + " unpaired references");
		
		// pair up indications
		PairMap<IIndicationStructure> indPairs = pairIndications(modelOne, modelTwo, refPairs);
		
		// process reference pairs
		HasSegmentAligner<IConstituent> aligner = new HasSegmentAligner<IConstituent>(tokenPairs);
		for(Entry<IIndicationStructure, IIndicationStructure> pair : indPairs.entrySet()){
			processPair(pair.getKey(), pair.getValue(), aligner, builder);
			processCopulars(pair.getKey(), pair.getValue(), builder);
		}

		return builder.toResult(one, two);
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param modelOne
	 * @param modelTwo
	 * @param builder
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected PairMap<IDesc> pairTokens(IStoryModel modelOne, IStoryModel modelTwo, AgreeResultBuilder builder){
		
		// token lists
		List<IDesc> tokensOne = new ArrayList<IDesc>(modelOne.getData().getDescriptions(TokenRep.getInstance()));
		List<IDesc> tokensTwo = new ArrayList<IDesc>(modelTwo.getData().getDescriptions(TokenRep.getInstance()));
		
		// record data
		builder.tokensOne = tokensOne.size();
		builder.tokensTwo = tokensTwo.size();
		
		// align them
		TokenAligner aligner = new TokenAligner();
		List<Pair<IDesc, IDesc>> tokenPairs = aligner.align(tokensOne, tokensTwo);
		
		// dump the pairs into the map
		PairMap<IDesc> result = new PairMap<IDesc>(tokenPairs);
		
		// record data
		builder.tokenPairs = result.size();
		builder.tokensUnpairedOne = result.oneUnpaired.size();
		builder.tokensUnpairedTwo = result.twoUnpaired.size();
		
		return result;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param modelOne
	 * @param modelTwo
	 * @param tokenMap
	 * @param builder
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected PairMap<IReference> pairReferences(IStoryModel modelOne, IStoryModel modelTwo, PairMap<IDesc> tokenMap, AgreeResultBuilder builder){
		
		// get references
		List<IReference> oneRefs = extractReferences(modelOne);
		List<IReference> twoRefs = extractReferences(modelTwo);
		
		// record data
		builder.refsOne = oneRefs.size();
		builder.refsTwo = twoRefs.size();
		
		// align them
		HasSegmentAligner<IReference> aligner = new HasSegmentAligner<IReference>(tokenMap);
		List<Pair<IReference, IReference>> refPairs = aligner.align(oneRefs, twoRefs);
		
//		for(Pair<?,?> pair : refPairs) System.out.println(pair);
		
		// dump the pairs into the map
		PairMap<IReference> result = new PairMap<IReference>(refPairs);
		
		// record data
		builder.refPairs = result.size();
		builder.refsUnpairedOne = result.oneUnpaired.size();
		builder.refsUnpairedTwo = result.twoUnpaired.size();
		
		return result;
	}

	/**
	 * TODO: Write comment
	 *
	 * @param modelOne
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected List<IReference> extractReferences(IStoryModel model) {
		IHasPositionSet<IReference> result = new HasPositionSet<IReference>();
		for(IDesc desc : model.getData().getDescriptions(ReferentRep.getInstance())){
			result.addAll(((IReferent)desc.getData()).getReferences());
		}
		return new ArrayList<IReference>(result);
	}
	
	
	/**
	 * TODO: Write comment
	 *
	 * @param modelOne
	 * @param modelTwo
	 * @param refPairs
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected PairMap<IIndicationStructure> pairIndications(IStoryModel modelOne, IStoryModel modelTwo, PairMap<IReference> refPairs){
		
		IDescSet indOneDescs = modelOne.getData().getDescriptions(IndicationStructureRep.getInstance());
		IDescSet indTwoDescs = modelTwo.getData().getDescriptions(IndicationStructureRep.getInstance());
		
		// construct ref->ind map for model two
		Map<IReference, IIndicationStructure> modelTwoMap = new HashMap<IReference, IIndicationStructure>(indTwoDescs.size());
		IIndicationStructure indTwo;
		for(IDesc d : indTwoDescs){
			indTwo = (IIndicationStructure)d.getData();
			modelTwoMap.put(indTwo.getReference(), indTwo);
		}
		
		// result for caller
		PairMap<IIndicationStructure> result = new PairMap<IIndicationStructure>();
		
		// get pairs
		IIndicationStructure indOne;
		IReference refTwo;
		for(IDesc d : indOneDescs){
			indOne = (IIndicationStructure)d.getData();
			refTwo = refPairs.get(indOne.getReference());
			if(refTwo == null){
				result.oneUnpaired.add(indOne);
			} else {
				indTwo = modelTwoMap.remove(refTwo);
				if(indTwo != null) result.put(indOne, indTwo);
			}
		}
		
		// make sure to add unpaired twos
		result.twoUnpaired.addAll(modelTwoMap.values());
		
		return result;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param refPairs
	 * @param builder
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void processPair(IIndicationStructure one, IIndicationStructure two, Aligner<IConstituent, IConstituent> aligner, AgreeResultBuilder builder){

		List<IConstituent> csOne = extractConstituents(one);
		List<IConstituent> csTwo = extractConstituents(two);
		
		builder.constsOne += csOne.size();
		builder.constsTwo += csTwo.size();
		
		List<Pair<IConstituent, IConstituent>> pairs = aligner.align(csOne, csTwo);
		
//		for(Pair<?,?> pair : pairs) System.out.println(pair);
//		System.out.println();
		
		boolean isOneDesc = false;
		boolean isTwoDesc = false;
		
		boolean isOne, isTwo;
		for(Pair<IConstituent, IConstituent> pair : pairs){
			
			// a unmatched
			if(pair.a != null && pair.b == null){
				builder.constsUnpairedOne++;
			}
			// b unmatched
			else if(pair.a == null && pair.b != null){
				builder.constsUnpairedTwo++;
			}
			// matched
			else {
				builder.constPairs++;
				
				// process type
				isOne = pair.a instanceof INucleus;
				isTwo = pair.b instanceof INucleus;
				if(isOne && isTwo){
					builder.constNucNuc++;
				} else if(isOne && !isTwo){
					builder.constNucMod++;
				} else if(!isOne && isTwo){
					builder.constModNuc++;
				} else {
					builder.constModMod++;
				}
				
				// process constituents
				isOne = pair.a.getType() == IndicationType.DESCRIPTIVE;
				isTwo = pair.b.getType() == IndicationType.DESCRIPTIVE;
				isOneDesc |= isOne;
				isTwoDesc |= isTwo;
				if(isOne && isTwo){
					builder.constDescDesc++;
				} else if(isOne && !isTwo){
					builder.constDescDist++;
				} else if(!isOne && isTwo){
					builder.constDistDesc++;
				} else {
					builder.constDistDist++;
				}
			}
		}
		
		// process overall reference tag
		if(!one.isCopularPredicate() && !two.isCopularPredicate()){
			if(isOneDesc && isTwoDesc){
				builder.refDescDesc++;
			} else if(isOneDesc && !isTwoDesc){
				builder.refDescDist++;
			} else if(!isOneDesc && isTwoDesc){
				builder.refDistDesc++;
			} else {
				builder.refDistDist++;
			}
		}
	}
	
	
	/**
	 * TODO: Write comment
	 *
	 * @param is
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected List<IConstituent> extractConstituents(IIndicationStructure is){
		IHasPositionSet<IConstituent> result = new HasPositionSet<IConstituent>();
		result.addAll(is.getNuclei());
		result.addAll(is.getModifiers());
		return new ArrayList<IConstituent>(result);
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param one
	 * @param two
	 * @param builder
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void processCopulars(IIndicationStructure one, IIndicationStructure two,AgreeResultBuilder builder){
		if(one.isCopularPredicate() && two.isCopularPredicate()){
			builder.copularYY++;
		} else if(one.isCopularPredicate() && !two.isCopularPredicate()){
			builder.copularYN++;
		} else if(!one.isCopularPredicate() && two.isCopularPredicate()){
			builder.copularNY++;
		} else {
			builder.copularNN++;
		}
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param results
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void calculateOverall(IResource one, IResource two, List<IResult> resultList, IResource test){
		
		// extract IAgreeResult objects
		IPath testPath = test.getProjectRelativePath();
		List<IAgreeResult> results = new LinkedList<IAgreeResult>();
		for(IResult r : resultList){
			if(r instanceof IAgreeResult && testPath.isPrefixOf(r.getTarget().getProjectRelativePath())){
				results.add((IAgreeResult)r);
			}
		}
		
		int a,b,c,d;
		OverallAgreeBuilder builder = new OverallAgreeBuilder();
		
		// calculate token f-measure
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getTokenPairs();
			b += result.getTokensUnpairedOne();
			c += result.getTokensUnpairedTwo();
		}
		builder.tokensF = fmeasure(a, b, c);
		
		// calculate references f-measure
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getRefPairs();
			b += result.getRefsUnpairedOne();
			c += result.getRefsUnpairedTwo();
		}
		builder.refsF = fmeasure(a, b, c);
		
		// calculate constituents f-measure
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getConstPairs();
			b += result.getConstsUnpairedOne();
			c += result.getConstsUnpairedTwo();
		}
		builder.constsF = fmeasure(a, b, c);
		
		// calculate copular kappa
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getCopularYY();
			b += result.getCopularYN();
			c += result.getCopularNY();
			d += result.getCopularNN();
		}
		builder.copularK = kappa(a, b, c, d);
		
		// calculate constituents label kappa
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getConstNucNuc();
			b += result.getConstNucMod();
			c += result.getConstModNuc();
			d += result.getConstModMod();
		}
		builder.constsK = kappa(a, b, c, d);
		
		// calculate constituents function kappa
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getConstDescDesc();
			b += result.getConstDescDist();
			c += result.getConstDistDesc();
			d += result.getConstDistDist();
		}
		builder.constsFuncK = kappa(a, b, c, d);
		
		// calculate references function kappa
		a = 0; b = 0; c = 0; d = 0;
		for(IAgreeResult result : results){
			a += result.getRefsDescDesc();
			b += result.getRefsDescDist();
			c += result.getRefsDistDesc();
			d += result.getRefsDistDist();
		}
		builder.refsFuncK = kappa(a, b, c, d);
		
		resultList.add(builder.toResult(one, two, (one == test ? "total" : test.getName())));
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param a both agree positive
	 * @param b first positive, second negative
	 * @param c first negative, second positive
	 * @param d both agree negative
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public static double kappa(double a, double b, double c, double d){
		if(a == 0 && b == 0 && c == 0 && d == 0) return 0;
		return 2*(a*d-b*c)/((a+c)*(c+d)+(a+b)*(b+d));
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param a both agree positive
	 * @param b first positive, second negative
	 * @param c first negative, second positive
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public static double fmeasure(double a, double b, double c){
		if(a == 0 && b == 0 && c == 0) return 0;
		return 2*a/(2*a+b+c);
	}

	/** 
	 * TODO: Write comment
	 *
	 * @author M.A. Finlayson
	 * @version $Rev$, $LastChangedDate$
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected class PairMap<T> extends LinkedHashMap<T, T> {

		private static final long serialVersionUID = 4768700554299082105L;
		
		private final Set<T> oneUnpaired;
		private final Set<T> twoUnpaired;
		
		public PairMap(){
			oneUnpaired = new LinkedHashSet<T>();
			twoUnpaired = new LinkedHashSet<T>();
		}
		
		public PairMap(List<Pair<T, T>> pairs){
			this();
			for(Pair<T, T> pair : pairs){
				if(pair.a == null){
					twoUnpaired.add(pair.b);
				} else if (pair.b == null){
					oneUnpaired.add(pair.a);
				} else {
					put(pair.a, pair.b);
				}
			}
		}
		
		public Set<T> getOneUnpaired(){
			return oneUnpaired;
		}
		
		public Set<T> getTwoUnpaired(){
			return twoUnpaired;
		}
	}
	
	/** 
	 * TODO: Write comment
	 *
	 * @author M.A. Finlayson
	 * @version $Rev$, $LastChangedDate$
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected class TokenAligner extends Aligner<IDesc, IDesc> {

		/* 
		 * (non-Javadoc) @see nil.ucm.indications2.ui.agreement.Aligner#sim(java.lang.Object, java.lang.Object)
		 */
		@Override
		public int sim(IDesc a, IDesc b) {
			IToken aToken = (IToken)a.getData();
			IToken bToken = (IToken)b.getData();
			return aToken.getSurface().equals(bToken.getSurface()) ? 1 : 0;
		}
		
	}
	
	/** 
	 * TODO: Write comment
	 *
	 * @author M.A. Finlayson
	 * @version $Rev$, $LastChangedDate$
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected class HasSegmentAligner<H extends IHasSegments> extends Aligner<H,H> {
		
		private final Map<IDesc, IDesc> tokenMap;
		
		public HasSegmentAligner(Map<IDesc, IDesc> tokenMap){
			if(tokenMap == null) throw new NullPointerException();
			this.tokenMap = tokenMap;
		}

		/* 
		 * (non-Javadoc) @see nil.ucm.indications2.ui.agreement.Aligner#sim(java.lang.Object, java.lang.Object)
		 */
		@Override
		public int sim(H a, H b) {
			Set<IDesc> aTokens = Segment.extractSegmentDescriptions(a);
			Set<IDesc> bTokens = Segment.extractSegmentDescriptions(b);
			for(Iterator<IDesc> i = aTokens.iterator(); i.hasNext(); ){
				if(!bTokens.contains(tokenMap.get(i.next()))) i.remove();
			}
			return aTokens.size();
		}
		
	}
	
}

