/* Filename: EnrichedReference.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 22/09/2009
 */
package nil.ucm.indications2.core.rep;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.Referent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.desc.Data;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.IStructuredData;
import edu.mit.story.core.desc.SegmentMapper;
import edu.mit.story.core.mappers.BlockMapper;
import edu.mit.story.core.mappers.IValueMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.PositionUtils;

/** 
 *   Implementation of the IIndicationStructure to represent an indication to a referent.
 * An indication structure is a special type of reference that contains information about the 
 * nuclei of the reference, their modifiers, and the type of the reference (distinctive, descriptive or
 * other).
 *
 * @author Raquel Hervas
 * @version 1.0, (Dec. 29, 2009)
 * @since nil.ucm.reference.core 1.0.0
 */
public class IndicationStructure implements IIndicationStructure {
	
	/** The referent that is being mentioned in the indication */
	private final IDesc referent;
	
	/** An indication always corresponds to a reference */
	private final IReference reference;
	
	/** List of nucleus in this indication */
	private final List<INucleus> nuclei;

	/** List of modifiers in this indication */
	private final List<IModifier> modifiers;
	
	private final boolean isCopularPredicate;
	
	public IndicationStructure(IDesc referent, int refID, List<? extends INucleus> nuclei, List<? extends IModifier> modifiers, boolean isCopularPredicate){
		if(referent == null) throw new NullPointerException();
		if(!ReferentRep.getInstance().isType(referent)) throw new IllegalArgumentException();
		if(nuclei.isEmpty()) throw new IllegalArgumentException();
		for(INucleus n : nuclei){
			if(n == null) throw new NullPointerException();
			n.setParent(this);
		}
		for(IModifier m : modifiers){
			if(m == null) throw new NullPointerException();
			m.setParent(this);
		}
		
		IReference reference = ((IReferent)referent.getData()).getReference(refID);
		if(reference == null) throw new NullPointerException();
		
		this.referent = referent;
		this.reference = reference;
		this.nuclei = Collections.unmodifiableList(new ArrayList<INucleus>(nuclei));
		this.modifiers = modifiers.isEmpty() ? Collections.<IModifier>emptyList() : Collections.unmodifiableList(new ArrayList<IModifier>(modifiers));
		this.isCopularPredicate = isCopularPredicate;
	}
	

	/* 
	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getName()
	 */
	
	public String getName() {
		return reference.getDisplayText();
	}
	
	/* 
	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getModifiers()
	 */
	
	public List<IModifier> getModifiers() {
		return modifiers;
	}
	

	/* 
	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getNuclei()
	 */
	
	public List<INucleus> getNuclei() {
		return nuclei;
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getReference()
	 */
	
	public IReference getReference() {
		return reference;
	}

//	/* 
//	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getType()
//	 */
//	@Override
//	public IndicationType getType() {
//		return type;
//	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
	 */
	
	public int getLength() {
		return reference.getLength();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
	 */
	
	public int getOffset() {
		return reference.getOffset();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
	 */
	
	public int getRightOffset() {
		return reference.getRightOffset();
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.reference.core.rep.reference.IEnrichedReference#getReferentDescription()
	 */
	
	public IDesc getReferentDescription() {
		return referent;
	}
	
	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.core.rep.IIndicationStructure#isCopularPredicate()
	 */
	public boolean isCopularPredicate() {
		return isCopularPredicate;
	}


	public static String serialize(IIndicationStructure is){
		
		//There are five blocks in the description
		List<String> blocks = new ArrayList<String>(5);
				
		// First field: display text
		IReference reference = is.getReference();
		IDesc referentDesc = is.getReferentDescription();
		blocks.add(reference.getDisplayText());
		
		// Second field: referent and reference ids
		blocks.add(referentDesc.getID() + ":" + reference.getID());
		
		// Third field: nuclei segments
		StringBuilder sb = new StringBuilder(64);
		for (INucleus n : is.getNuclei()) {
			sb.append(SegmentMapper.serializeStatic(n.getSegments()));
			sb.append(":");
			sb.append(getTypeString(n.getType()));
			sb.append(";");
		}
		sb.deleteCharAt(sb.length()-1);
		blocks.add(sb.toString());
		
		// Fourth field: modifiers segments
		sb = new StringBuilder(64);
		if (is.getModifiers().size() > 0) {
			for (IModifier m : is.getModifiers()) {
				sb.append(SegmentMapper.serializeStatic(m.getSegments()));
				sb.append(":");
				sb.append(getTypeString(m.getType()));
				sb.append(";");
			}
			sb.deleteCharAt(sb.length() - 1);
		}
		blocks.add(sb.toString());
		
		// Fifth field: copular flag
		if(is.isCopularPredicate()) blocks.add("copula");
				
		return BlockMapper.PIPE.serialize(blocks);
	}
	
	private static String getTypeString(IndicationType type) {
		
		String typeStr;
		switch (type)
		{
			case DESCRIPTIVE : typeStr = "DE"; break;
			case DISTINCTIVE : typeStr = "DI"; break;
			case OTHER : typeStr = "OT"; break;
			default : typeStr = "UN"; break;
		}
		
		return typeStr;
	}
	
	public static IndicationStructure reconstitute(String input, IStoryData data){
		
		List<String> blocks = BlockMapper.PIPE.reconstitute(input);
		IValueMapper<ISegment> mapper = new SegmentMapper(TokenRep.getInstance(), data);
		
		int idx;
				
		//There are five blocks in the description: text, referent:reference ids, nuclei, modifiers and type
		//The text is not read because it is not used. It is always the text associated with the reference.
		//String text = blocks.get(0).trim();
		
		//referent:reference ids (second block)
		String blockRef = blocks.get(1);
		idx = blockRef.indexOf(':');
		int idReferent = Integer.parseInt(blockRef.substring(0,idx));
		int idReference = Integer.parseInt(blockRef.substring(idx+1,blockRef.length()));
		
		IDesc referentDesc = data.getDescription(idReferent);
		
		//List of nuclei (third block)
		String[] info;
		String blockNuc = blocks.get(2);
		String[] nuc = blockNuc.split(";");
		List<INucleus> nuclei = new ArrayList<INucleus>();
		for (int i=0; i<nuc.length; i++) {
			//The segments and the type of the nucleus are separated by ':'
			info = nuc[i].split(":");
			nuclei.add(new Nucleus(mapper.reconstitute(info[0]), getTypeFromString(info[1])));
		}
		
		//List of modifiers (fourth block)
		String blockMod = blocks.get(3);
		String[] mods = blockMod.split(";");
		List<IModifier> modifiers = new ArrayList<IModifier>();
		if (!blockMod.equalsIgnoreCase("")){
			for (int i=0; i<mods.length; i++) {
				//The segments and the type of the modifier are separated by ':'
				info = mods[i].split(":");
				modifiers.add(new Modifier(mapper.reconstitute(info[0]), getTypeFromString(info[1])));
			}
		}
		
		// copula flag (fifth block)
		boolean isCopula = false;
		if(blocks.size() > 4) isCopula = blocks.get(4).equals("copula");
	
		return new IndicationStructure(referentDesc, idReference, nuclei, modifiers, isCopula);
	}
	
	private static IndicationType getTypeFromString(String str) {
		IndicationType type;
		if (str.equalsIgnoreCase("DI")) type = IndicationType.DISTINCTIVE;
		else if (str.equalsIgnoreCase("DE")) type = IndicationType.DESCRIPTIVE;
		else if (str.equalsIgnoreCase("OT")) type = IndicationType.OTHER;
		else type = IndicationType.UNKNOWN;
		
		return type;
	}


	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition()
	 */
	public IHasPosition calculatePosition() {
		return PositionUtils.makePosition(reference);
	}


	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}


	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#recalculate(edu.mit.story.core.model.IStoryModel)
	 */
	
	public IStructuredData recalculate(IDesc container, IStoryModel model) {
		
		//See if referent or reference have changed
		IDesc newReferent = model.getData().getDescriptions(ReferentRep.getInstance()).getDescription(referent.getID());
	
		//If the referent does not longer exists, the indication structure must also be deleted
		if (newReferent == null) return null;
		
		//If there is no reference, the indication is not valid anymore
		if(((Referent)newReferent.getData()).getReference(reference.getID()) == null) return null;
		
		//The Indication Structure has to be updated if the referent or the reference have changed
		boolean changed = (newReferent != referent);
		
		ISegment newSeg;
		Set<ISegment> newSegs;

		//We check for changes in the nuclei
		List<INucleus> newNuclei = new ArrayList<INucleus>();
		for(INucleus n : nuclei){
			newSegs = new HashSet<ISegment>(n.getSegments().size());
			for(ISegment oldSeg : n.getSegments()){
				newSeg = (ISegment)oldSeg.recalculate(container, model);
				if(newSeg != null) newSegs.add(newSeg);
				changed |= (newSeg != oldSeg);
			}
			if(!newSegs.isEmpty()) newNuclei.add(new Nucleus(newSegs, n.getType()));
		}
		//It is not possible to have an indication structure without a nucleus
		if (newNuclei.isEmpty()) return null;
		
		//We check for changes in the modifiers
		List<IModifier> newModifiers = new ArrayList<IModifier>();
		for(IModifier m : modifiers){
			newSegs = new HashSet<ISegment>(m.getSegments().size());
			for(ISegment oldSeg : m.getSegments()){
				newSeg = (ISegment)oldSeg.recalculate(container, model);
				if(newSeg != null) newSegs.add(newSeg);
				changed |= (newSeg != oldSeg);
			}
			if(!newSegs.isEmpty()) newModifiers.add(new Modifier(newSegs, m.getType()));
		}
		
		//Something has changed but the indication is still correct. The id of the reference is still correct because the set of segments is not empty
		return new IndicationStructure(newReferent, reference.getID(), newNuclei, newModifiers, isCopularPredicate);

	}
	
	public String toString(){
		return IndicationStructureRep.getInstance().serialize(this);
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		if(this == tgtData)
			return true;
		if(!IIndicationStructure.class.isAssignableFrom(tgtData.getClass()))
			return false;
		IIndicationStructure tgtInd = (IIndicationStructure)tgtData;
		
		if(!reference.equals(tgtInd.getReference()))
			return false;
		if(!Data.equals(nuclei, tgtInd.getNuclei(), model))
			return false;
		if(!Data.equals(modifiers, tgtInd.getModifiers(), model))
			return false;
		
		return true;
	}


//	/* 
//	 * (non-Javadoc) @see nil.ucm.indications.core.rep.IIndicationStructure#setType(nil.ucm.indications.core.rep.IndicationType)
//	 */
//	@Override
//	public void setType(IndicationType t) {
//		this.type = t;
//	}
}
