/* Filename: CountCalculator.java
 * Creator: M.A. Finlayson
 * Format: Java 2 v1.6.0
 * Date created: Feb 9, 2010
 */
package nil.ucm.indications2.ui.agreement;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import nil.ucm.indications2.core.rep.IIndicationStructure;
import nil.ucm.indications2.core.rep.IModifier;
import nil.ucm.indications2.core.rep.INucleus;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.ui.agreement.CountResult.CountResultBuilder;
import nil.ucm.indications2.ui.agreement.TimeResult.TimeResultBuilder;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;

import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.parsing.core.rep.sentence.SentenceRep;
import edu.mit.parsing.core.rep.token.IToken;
import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.measure.IResult;
import edu.mit.story.core.measure.Result;
import edu.mit.story.core.measure.StoryFileExtractor;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.StoryModelImporter;
import edu.mit.story.core.rep.IRep;
import edu.mit.story.core.util.Debug;

/** 
 * TODO: Write comment
 *
 * @author M.A. Finlayson
 * @version $Rev$, $LastChangedDate$
 * @since nil.ucm.indications2.ui 1.0.0
 */
public class CountCalculator {
	
//	private TimeCalculator timeCalc = new TimeCalculator(IndicationStructureRep.getInstance()); 

	/**
	 * Calculates the counts of various items over the resource specified. The
	 * resource may be a project, folder, or file. If a project or folder, the
	 * method will recursively discover all the story files and process them.
	 * 
	 * @param r
	 *            the resource to be processed
	 * @return a list of results for
	 * @throws NullPointerException
	 *             if the specified resource is <code>null</code>
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public List<IResult> calculate(IResource r, IProgressMonitor pm){
		if(r == null) throw new NullPointerException();
		
		List<IResult> results = new LinkedList<IResult>();
		List<IFile> files = StoryFileExtractor.getInstance().getTargets(r, results);
		
		// collect list of folders by which to generate overalls
		Set<IResource> tests = new HashSet<IResource>();
		
		// do counting over files we found
		pm.beginTask("Calculating Counts", files.size()+1);
		IStoryModel model;
		String msg;
		for(IFile file : files){
			msg = "Processing " + file.getName();
			pm.subTask(msg);
			Debug.err(msg);
			tests.add(file.getParent());
			model = StoryModelImporter.extractModel(file, results);
			if(model != null) results.add(calculate(file, model));
			pm.worked(1);
			if(pm.isCanceled()) return results;
		}
		
		// dump to list
		List<IResource> testList = new ArrayList<IResource>(tests.size() + 1);
		tests.remove(r);
		testList.addAll(tests);
		testList.add(r);
		
		// calculate final kappas+f-measures
		msg = "Calculating overall agreement";
		pm.subTask(msg);
		List<ICountResult> overall = new ArrayList<ICountResult>(testList.size());
		ICountResult result;
		for(IResource test : testList){
			result = calculateOverall(r, results, test);
			if(result != null)
				overall.add(result);
		}
		results.addAll(overall);
		pm.worked(1);
		pm.done();

		return results;
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @throws NullPointerException if the model is <code>null</code>
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	public IResult calculate(IFile file, IStoryModel model){
		if(model == null) throw new NullPointerException();
		try{
			CountResultBuilder builder = new CountResultBuilder();
			countWords(model, builder);
			countSentences(model, builder);
			countReferents(model, builder);
			countIndications(model, builder);
			
//			builder.timeResult = timeCalc.calculate(file, model);
			return builder.toResult(file);
		} catch(Exception e){
			e.printStackTrace();
			return new Result(file, "Error counting file", e);
		}
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void countWords(IStoryModel model, CountResultBuilder counts){
		
		IRep rep = TokenRep.getInstance();
		if(!model.supports(rep)) throw new IllegalArgumentException("Model doesn't support " + rep.getName());
		
		// don't count words containing all
		// non-alphanumeric characters, or "'s" as words
		IToken token;
		for(IDesc desc : model.getData().getDescriptions(rep)){
			token = (IToken)desc.getData();
			if(isWord(token.getSurface())) counts.words++;
		}
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void countSentences(IStoryModel model, CountResultBuilder counts){
		IRep rep = SentenceRep.getInstance();
		if(!model.supports(rep)) throw new IllegalArgumentException("Model doesn't support " + rep.getName());
		counts.sentences += model.getData().getDescriptions(rep).size();
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void countReferents(IStoryModel model, CountResultBuilder counts){
				
		IRep rep = ReferentRep.getInstance();
		if(!model.supports(rep)) throw new IllegalArgumentException("Model doesn't support " + rep.getName());
		
		// don't count words containing all
		// non-alphanumeric characters, or "'s" as words
		IReferent ref;
		int numRefs;
		for(IDesc desc : model.getData().getDescriptions(rep)){
			ref = (IReferent)desc.getData();
			counts.referents++;
			numRefs = ref.getReferences().size();
			counts.references += numRefs;
			counts.maxReferences = Math.max(counts.maxReferences, numRefs);
		}
		
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected void countIndications(IStoryModel model, CountResultBuilder counts){
				
		IRep rep = IndicationStructureRep.getInstance();
		if(!model.supports(rep)) throw new IllegalArgumentException("Model doesn't support " + rep.getName());
		
		// don't count words containing all
		// non-alphanumeric characters, or "'s" as words
		IIndicationStructure is;
		boolean descriptive;
		for(IDesc desc : model.getData().getDescriptions(rep)){
			is = (IIndicationStructure)desc.getData();
			counts.indications++;
			descriptive = false;
			
			// if copular, ignore
			if(is.isCopularPredicate()){
				counts.copReferences++;
				continue;
			}
			
			// count nuclei
			for(INucleus n : is.getNuclei()){
				counts.nuclei++;
				switch(n.getType()){
				case DISTINCTIVE:
					counts.distNuclei++;
					break;
				case DESCRIPTIVE:
					counts.descNuclei++;
					descriptive = true;
					break;
				}
			}
			counts.maxNuclei = Math.max(counts.maxNuclei, is.getNuclei().size());
			
			// count modifiers
			for(IModifier m : is.getModifiers()){
				counts.modifiers++;
				switch(m.getType()){
				case DISTINCTIVE:
					counts.distModifiers++;
					break;
				case DESCRIPTIVE:
					counts.descModifiers++;
					descriptive = true;
					break;
				}
			}
			counts.maxModifiers = Math.max(counts.maxModifiers, is.getModifiers().size());
			if(descriptive){
				counts.descReferences++;
			} else {
				counts.distReferences++;
			}
		}
		
	}
	
	/**
	 * TODO: Write comment
	 *
	 * @param results
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected ICountResult calculateOverall(IResource r, List<IResult> resultList, IResource test){
		
		// extract ICountResult objects
		IPath testPath = test.getProjectRelativePath();
		List<ICountResult> results = new LinkedList<ICountResult>();
		for(IResult result : resultList)
			if(result instanceof ICountResult && testPath.isPrefixOf(result.getTarget().getProjectRelativePath()))
					results.add((ICountResult)result);
		
		CountResultBuilder builder = new CountResultBuilder();
		TimeResultBuilder timeBuilder = new TimeResultBuilder();
		
		timeBuilder.start = Long.MAX_VALUE;
		timeBuilder.end = Long.MIN_VALUE;
		
		for(ICountResult result : results){
			builder.words += result.getWords();
			builder.sentences += result.getSentences();
			builder.referents += result.getReferents();
			builder.references += result.getReferences();
			builder.maxReferences = Math.max(result.getMaxReferences(), builder.maxReferences);
			builder.distReferences += result.getDistinctiveRefs();
			builder.descReferences += result.getDescriptiveRefs();
			builder.indications += result.getIndications();
			builder.nuclei += result.getNuclei();
			builder.maxNuclei = Math.max(result.getMaxNuclei(), builder.maxNuclei);
			builder.distNuclei += result.getDistinctiveNucs();
			builder.descNuclei += result.getDescriptiveNucs();
			builder.modifiers += result.getModifiers();
			builder.maxModifiers = Math.max(result.getMaxModifiers(), builder.maxModifiers);
			builder.distModifiers += result.getDistinctiveMods();
			builder.descModifiers += result.getDescriptiveMods();
			
			timeBuilder.start = Math.min(timeBuilder.start, result.getTimeResult().getStart());
			timeBuilder.end = Math.max(timeBuilder.end, result.getTimeResult().getEnd());
			timeBuilder.total += result.getTimeResult().getTotal();
			timeBuilder.rep = result.getTimeResult().getRep();
		}
		
		builder.timeResult = timeBuilder.toResult(test);
		return builder.toResult(test);
	}

	
	/**
	 * TODO: Write comment
	 *
	 * @param token
	 * @return
	 * @since nil.ucm.indications2.ui 1.0.0
	 */
	protected static boolean isWord(String token){
		if(token.equalsIgnoreCase("'s")) return false;
		for(int i = 0; i < token.length(); i++){
			if(Character.isLetterOrDigit(token.charAt(i))) return true;
		}
		return false;
	}
	
}
