/* Filename: ReferentModel.java
 * Creator: M.A. Finlayson
 * Format: Java 2 v1.6.0
 * Date created: Sep 8, 2009
 */
package edu.mit.discourse.core.rep.referent.model;

import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.eclipse.jface.dialogs.IMessageProvider;

import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.Referent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.datamodel.AbstractViewModel;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.IStructuredData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.model.change.IModelListener;
import edu.mit.story.core.model.change.StoryChangeEvent;
import edu.mit.story.core.notify.MessageProviderManager;
import edu.mit.story.core.position.HasPositionSet;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.validation.Message;

/** 
 * TODO: Write comment
 *
 * @author M.A. Finlayson
 * @version $Rev$, $LastChangedDate$
 * @since edu.mit.discourse.ui 1.0.0
 */
public class ReferentModel extends AbstractViewModel implements IReferentModel, ChangeListener, IModelListener {
	
	public static final IMessageProvider noReferences = new Message("The referent must have at least one reference", IMessageProvider.ERROR);
	
	private final IStoryModel model;

	private String name = null;
	private final Map<IReferenceModel, Integer> idMap = new IdentityHashMap<IReferenceModel, Integer>(); 
	private final Map<IReferenceModel, Integer> fixedIDMap = new IdentityHashMap<IReferenceModel, Integer>();
//	private final IHasPositionSet<IReferentPropertyModel> properties = new HasPositionSet<IReferentPropertyModel>();
//	private final IHasPositionSet<IReferentPropertyModel> extProperties = new ImmutableHasPositionSet<IReferentPropertyModel>(properties);
	private int minID = 0;
	private IDesc loaded = null;
	private MessageProviderManager manager;
	
	/**
	 * TODO: Write comment
	 *
	 * @param model
	 * @since edu.mit.discourse.ui 1.0.0
	 */
	public ReferentModel(IStoryModel model) {
		if(model == null) throw new NullPointerException();
		this.model = model;
		this.model.addModelListener(this);	
		this.manager = new MessageProviderManager();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReferent#getName()
	 */
	
	public String getName() {
		return name == null ? "" : name;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#getReference(int)
	 */
	
	public IReferenceModel getReference(int id) {
		for(Entry<IReferenceModel, Integer> entry : idMap.entrySet()){
			if(entry.getValue() == id) return entry.getKey();
		}
		return null;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#getID(edu.mit.discourse.ui.models.IReferenceModel)
	 */
	
	public int getID(IReferenceModel ref) {
		return idMap.get(ref);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReferent#getReferences()
	 */
	
	public IHasPositionSet<IReference> getReferences() {
		return new HasPositionSet<IReference>(idMap.keySet());
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#addReference(edu.mit.discourse.ui.models.IReferenceModel)
	 */
	
	public void addReference(IReferenceModel ref) {
		ref.setParent(this);
		manager.add(ref);
		
		// put new references in id map
		idMap.put(ref, null);
		renumberReferences();

		// stimulate update
		setName(name);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#removeReference(int)
	 */
	
	public IReferenceModel removeReference(int id) {
		IReferenceModel ref = getReference(id);
		internalRemove(ref);
		setOutOfSync();
		notifyChangeListeners();
		return ref;
		
	}
	
	protected void internalRemove(IReferenceModel ref){
		idMap.remove(ref);
		fixedIDMap.remove(ref);
		manager.remove(ref);
		ref.setParent(null);
	}
	
	protected void renumberReferences(){
		int id = minID;
		Integer fixedID;
		for(IReferenceModel ref : new HasPositionSet<IReferenceModel>(idMap.keySet())){
			fixedID = fixedIDMap.get(ref);
			idMap.put(ref, fixedID == null ? id++ : fixedID);
		}
	}

//	/* 
//	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#getProperties()
//	 */
//	
//	public IHasPositionSet<IReferentPropertyModel> getProperties() {
//		return extProperties;
//	}
//
//	/* 
//	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#addProperty(edu.mit.discourse.ui.models.IReferentPropertyModel)
//	 */
//	public void addProperty(IReferentPropertyModel prop) {
//		if(properties.add(prop)) prop.addChangeListener(this);
//		setOutOfSync();
//		notifyChangeListeners();
//	}
//
//	/* 
//	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#removeProperty(edu.mit.discourse.ui.models.IReferentPropertyModel)
//	 */
//	
//	public void removeProperty(IReferentPropertyModel prop) {
//		if(properties.remove(prop)) prop.removeChangeListener(this);
//		setOutOfSync();
//		notifyChangeListeners();
//	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#setName(java.lang.String)
	 */
	
	public void setName(String name) {
		String newName = (name == null || name.trim().length() == 0) ? getDefaultName() : name;
		this.name = newName;
		setOutOfSync();
		notifyChangeListeners();
	}
	
	/**
	 * Returns the text of the first segment of the first reference, if there is one.
	 * Otherwise, returns <code>null</code>.
	 *
	 * @return
	 * @since edu.mit.discourse.ui 1.0.0
	 */
	protected String getDefaultName(){
		return idMap.isEmpty() ? null : getReferences().first().getSegments().first().getDisplayText();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.NotifyingMessageProvider#calculateMessage()
	 */
	@Override
	protected IMessageProvider calculateMessage() {
		return idMap.isEmpty() ? noReferences : manager.getFixedMessage();
	}

	protected void internalClear(){
		for(IReferenceModel ref : idMap.keySet()){
			manager.remove(ref);
			ref.setParent(null);
			ref.dispose();
		}
//		for(IReferentPropertyModel model : properties){
//			model.removeChangeListener(this);
//			model.dispose();
//		}
//		properties.clear();
		idMap.clear();
		fixedIDMap.clear();
		minID = 0;
		name = null;
		loaded = null;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.IClearable#isCleared()
	 */
	
	public boolean isCleared() {
		return loaded == null && name == null && idMap.isEmpty();
	}

	/* 
	 * (non-Javadoc) @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
	 */
	
	public void stateChanged(ChangeEvent e) {
		List<IReferenceModel> remove = new LinkedList<IReferenceModel>();
		for(IReferenceModel ref : idMap.keySet()){
			if(ref.getSegments().isEmpty()) remove.add(ref);
		}
		for(IReferenceModel ref : remove) internalRemove(ref);
		setOutOfSync();
		notifyChangeListeners();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.util.IDisposable#dispose()
	 */
	
	public void dispose() {
		this.model.removeModelListener(this);	
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
	 */
	
	public int getLength() {
		throw new UnsupportedOperationException();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
	 */
	
	public int getOffset() {
		throw new UnsupportedOperationException();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
	 */
	
	public int getRightOffset() {
		throw new UnsupportedOperationException();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return calculatePosition();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition()
	 */
	
	public IHasPosition calculatePosition() {
		throw new UnsupportedOperationException();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#recalculate(edu.mit.story.core.model.IStoryModel)
	 */
	
	public IStructuredData recalculate(IDesc container, IStoryModel model) {
		throw new UnsupportedOperationException();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		throw new UnsupportedOperationException();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#toReferent()
	 */
	
	public IReferent toReferent() {
		Map<Integer, IHasPositionSet<ISegment>> map = new HashMap<Integer, IHasPositionSet<ISegment>>();
		for(Entry<IReferenceModel, Integer> entry : idMap.entrySet()){
			if(map.containsKey(entry.getValue())) throw new IllegalStateException();
			map.put(entry.getValue(), entry.getKey().getSegments());
		}
		return new Referent(getName(), map, null);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#getLoaded()
	 */
	
	public IDesc getLoaded() {
		return loaded;
	}

	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.models.IReferentModel#load(edu.mit.story.core.desc.IDesc)
	 */
	
	public void load(IDesc desc) {
		internalClear();
		
		IReferent data = (IReferent)desc.getData();
		
		this.name = data.getName();
		IReferenceModel refModel;
		int newMinID = -1;
		for(IReference ref : data.getReferences()){
			refModel = new ReferenceModel(model);
			refModel.setSegments(ref.getSegments());
			idMap.put(refModel, ref.getID());
			fixedIDMap.put(refModel, ref.getID());
			refModel.setParent(this);
			manager.add(refModel);
			newMinID = Math.max(newMinID, ref.getID());
		}
		this.minID = newMinID+1;
		this.loaded = desc;
		setOutOfSync();
		notifyChangeListeners();
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.model.change.IModelListener#modelChanged(edu.mit.story.core.model.change.StoryChangeEvent)
	 */
	
	public void modelChanged(StoryChangeEvent e) {
		if(loaded == null) return;
		if(!e.affects(ReferentRep.getInstance())) return;
		
		IDesc newDesc = model.getData().getDescription(loaded.getID());
		
		// description is unchanged
		if(newDesc == loaded) return;
		
		// description was deleted
		if(newDesc == null){
			clear();
			return;
		} 
		
		// description needs to be reloaded
		load(newDesc);
	}

}
