/* Filename: Argument.java
 * Author: M. A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Dec 6, 2007
 */
package edu.mit.discourse.core.rep.relation;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.SegmentMapper;
import edu.mit.story.core.mappers.BlockMapper;
import edu.mit.story.core.mappers.IValueMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.HasPositionSet;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.position.SimplePosition;
import edu.mit.story.core.util.ArrayUtils;

/** One line description goes here...
 * More detail goes here...
 *
 * @author M.A. Finlayson
 * @since 1.5.0
 */
public class Argument extends Adjunct implements IArgument {

	/**
	 * List where adjuncts are stored; this list may not be {@code null}. It is
	 * initialized in the constructor.
	 *
	 * @since SW 1.0
	 */
	private final List<IAdjunct> adjuncts;
	
	/**
	 * Constructs a new argument out of the specified segments with no adjuncts.
	 * 
	 * @param segments
	 *            The collection of segments which comprise this argument. The
	 *            collection may be {@code null} or empty: this merely creates
	 *            an invalid argument (see {@link #isValid()}.
	 *
	 * @since SW 1.0
	 */
	public Argument(Collection<? extends ISegment> segments) {
		this(segments, null);
	}
	
	/**
	 * Constructs a new argument out of the specified segments, with the
	 * specified adjuncts.
	 * 
	 * @param segments
	 *            The collection of segments which comprise this argument. The
	 *            collection may be {@code null} or empty: this merely creates
	 *            an invalid argument (see {@link #isValid()}.
	 * @param adjuncts
	 *            The collection of adjunct to this argument. The collection may
	 *            be {@code null} or empty.
	 * @since SW 1.0
	 */
	public Argument(Collection<? extends ISegment> segments, Collection<? extends IAdjunct> adjuncts) {
		super(segments);
		this.adjuncts = ArrayUtils.makeNewUnmodifiableList(adjuncts);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see edu.mit.discourse.core.rep.relation.IArgument#getAdjuncts()
	 */
	public List<IAdjunct> getAdjuncts() {
		return adjuncts;
	}
	
	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.discourse.core.rep.relation.Adjunct#calculatePosition() 
	 */
	public IHasPosition calculatePosition() {
		IHasPositionSet<IHasPosition> result = new HasPositionSet<IHasPosition>();
		result.add(getSegments());
		for(IAdjunct adjunct : adjuncts) result.add(adjunct.getSegments());
		return new SimplePosition(result);
	}

	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.discourse.core.rep.relation.Adjunct#recalculate(edu.mit.story.core.model.IStoryModel) 
	 */
	public Argument recalculate(IDesc container, IStoryModel model) {
		
		List<IAdjunct> newAdjList = null;
		
		Adjunct newInstance = (Adjunct)super.recalculate(container, model);
		if(newInstance == null) return null; 
		if(newInstance != this) newAdjList = new ArrayList<IAdjunct>(adjuncts);
		
		IAdjunct newAdj;
		for(IAdjunct oldAdj : adjuncts){
			newAdj = (IAdjunct)oldAdj.recalculate(container, model);
			if(newAdj == oldAdj) continue;
			if(newAdjList == null) newAdjList = new ArrayList<IAdjunct>(adjuncts);
			newAdjList.set(newAdjList.indexOf(oldAdj), newAdj);
		}
		if(newAdjList == null) return this;
		return new Argument(newInstance.getSegments(), newAdjList); 
	}
	
	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.discourse.core.rep.relation.Adjunct#clone() 
	 */
	public Argument clone(){
		return new Argument(getSegments(), adjuncts);
	}
	
	public static String serialize(IArgument argument){
		String[] blocks = new String[argument.getAdjuncts().size()+1];
		blocks[0] = Adjunct.serialize(argument);
		int i = 1;
		for(IAdjunct adjunct : argument.getAdjuncts()) blocks[i++] = Adjunct.serialize(adjunct);
		return BlockMapper.SLASH_FORWARD.serialize(Arrays.asList(blocks));
	}
	
	public static Argument reconstitute(String description, IStoryData data){
		
		// split into blocks on slashes
		List<String> blocks = BlockMapper.SLASH_FORWARD.reconstitute(description);
		if(blocks.size() == 0) return null; // ill-formed
		
		// get segments for argument
		IValueMapper<ISegment> mapper = new SegmentMapper(TokenRep.getInstance(), data);
		List<ISegment> segments = mapper.reconstitute(blocks.get(0));
		if(segments.size() == 0) return null; // ill-formed
		
		// no adjuncts, so return
		if(blocks.size() == 1) return new Argument(segments);
		
		// extract adjuncts
		List<IAdjunct> adjuncts = new ArrayList<IAdjunct>();
		IAdjunct adjunct;
		for(int i = 0; i < blocks.size()-1; i++){
			adjunct = Adjunct.reconstitute(blocks.get(i+1), data);
			if(adjunct != null) adjuncts.add(adjunct);
		}
		
		return new Argument(segments, adjuncts);
	}
	
}
