/* Filename: AddEntityChange.java
 * Author: Mark A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Oct 23, 2007
 */
package edu.mit.discourse.core.rep.referent.change;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import edu.mit.discourse.core.DiscourseMessages;
import edu.mit.discourse.core.rep.referent.IReference;
import edu.mit.discourse.core.rep.referent.IReferent;
import edu.mit.discourse.core.rep.referent.Referent;
import edu.mit.discourse.core.rep.referent.ReferentRep;
import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.desc.Desc;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.Segment;
import edu.mit.story.core.meta.MetaDesc;
import edu.mit.story.core.meta.origin.IOrigin;
import edu.mit.story.core.meta.origin.Origin;
import edu.mit.story.core.meta.origin.OriginMetaRep;
import edu.mit.story.core.meta.timing.ITiming;
import edu.mit.story.core.meta.timing.TimingMetaRep;
import edu.mit.story.core.model.IMutableStoryModel;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.change.AbstractModelChange;
import edu.mit.story.core.model.change.IModelChange;
import edu.mit.story.core.model.change.ReplaceDescriptionChange;
import edu.mit.story.core.model.change.StoryChangeEvent;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.SimplePosition;


/** One line description goes here...
 * More detail goes here...
 *
 * @author Mark A. Finlayson
 * @version 1.00, (Oct 23, 2007)
 * @since 1.5.0
 */
public class AddReferenceChange extends AbstractModelChange {
	
	protected final long id;
	protected final Collection<? extends Collection<IHasPosition>> refs;
	protected final ITiming timing;
	
	public AddReferenceChange(IHasPosition reference, IDesc referent, ITiming timing){
		this(referent, Collections.singleton(reference), timing);
	}
	
	public AddReferenceChange(IDesc referent, Collection<? extends IHasPosition> reference, ITiming timing){
		this(Collections.singleton(reference), referent, timing);
	}
	
	public AddReferenceChange(Collection<? extends Collection<? extends IHasPosition>> references, IDesc referent, ITiming timing){
		super(DiscourseMessages.CHANGE_AddReference_name);
		
		// check arguments
		if(references.isEmpty()) throw new IllegalArgumentException();
		if(!ReferentRep.getInstance().isType(referent)) throw new IllegalArgumentException();
		
		// set fields
		Set<IHasPosition> inner;
		Set<Set<IHasPosition>> outer = new HashSet<Set<IHasPosition>>(references.size());
		for(Collection<? extends IHasPosition> reference : references){
			inner = new HashSet<IHasPosition>(reference.size());
			for(IHasPosition segment : reference) inner.add(new SimplePosition(segment));
			if(inner.isEmpty()) throw new IllegalArgumentException();
			outer.add(Collections.unmodifiableSet(inner));
		}
		
		this.id = referent.getID();
		this.refs = Collections.unmodifiableSet(outer);
		this.timing = timing;
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.provider.changes.AbstractSynchronizedProviderChange#synchronizedDoApply(java.lang.Object, edu.mit.story.core.provider.IDescriptionProvider) */
	@Override
	protected StoryChangeEvent doApply(Object source, IMutableStoryModel model) {
		
		// get entity description
		IDesc oldDesc = model.getData().getDescriptions(ReferentRep.getInstance()).getDescription(id);
		if(oldDesc == null) return null;
		
		// get the entity, check and see if it already has this segment
		IReferent oldData = (IReferent)oldDesc.getData();
		
		// construct the new reference map
		// if we can't find the reference to delete, don't do anything
		Map<Integer, Collection<? extends ISegment>> newRefMap = new HashMap<Integer, Collection<? extends ISegment>>(oldData.getReferences().size());
		for(IReference oldRef : oldData.getReferences()){
			newRefMap.put(oldRef.getID(), oldRef.getSegments());
		}
		
		// construct the new reference set 
		Set<Set<ISegment>> newRefSet = new HashSet<Set<ISegment>>();
		Set<ISegment> newSegSet;
		IStoryData data = model.getData();
		for(Collection<IHasPosition> newRef : refs){
			newSegSet = new HashSet<ISegment>(newRef.size());
			newRefSet.add(newSegSet);
			for(IHasPosition p : newRef) newSegSet.add(new Segment(p, TokenRep.getInstance(), data));
		}
		
		// add the new description
		IReferent newData = new Referent(oldData.getName(), newRefMap, newRefSet);
		IDesc newDesc = new Desc(oldDesc.getID(), ReferentRep.getInstance(), newData.calculatePosition(), newData, oldDesc.getMetaData());
		newDesc.getMetaData().addDesc(new MetaDesc<IOrigin>(OriginMetaRep.getInstance(), newDesc, Origin.USER_SPECIFIED));
		if(timing != null) 
			newDesc.getMetaData().addDesc(new MetaDesc<ITiming>(TimingMetaRep.getInstance(), newDesc, timing));

		IModelChange change = new ReplaceDescriptionChange(getName(), oldDesc, newDesc);
		return change.apply(source, model);
	}

}
