/* Filename: Entity.java
 * Author: Mark A. Finlayson
 * Format: Java 2 v1.5.0
 * Date created: Oct 23, 2007
 */
package edu.mit.discourse.core.rep.referent;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import edu.mit.parsing.core.rep.token.TokenRep;
import edu.mit.story.core.align.IAlignedStoryModel;
import edu.mit.story.core.desc.Data;
import edu.mit.story.core.desc.IData;
import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.IDescSet;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.ISegmentSet;
import edu.mit.story.core.desc.Segment;
import edu.mit.story.core.desc.SegmentMapper;
import edu.mit.story.core.desc.SegmentSet;
import edu.mit.story.core.mappers.BlockMapper;
import edu.mit.story.core.mappers.IValueMapper;
import edu.mit.story.core.model.IStoryData;
import edu.mit.story.core.model.IStoryModel;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.core.position.ImmutableHasPositionSet;
import edu.mit.story.core.position.PositionUtils;
import edu.mit.story.core.rep.IRep;


/** One line description goes here...
 * More detail goes here...
 *
 * @author Mark A. Finlayson
 * @version 1.00, (Oct 23, 2007)
 * @since 1.5.0
 */
public class Referent implements IReferent {

	// instance fields set on construction
	private final String name;
	private final ImmutableHasPositionSet<IReference> refs;
	
	public Referent(String name,  Map<Integer, ? extends Collection<? extends ISegment>> refsWithIDs, Collection<? extends Collection<? extends ISegment>> refsNoIDs) {
		if(name.trim().length() == 0) throw new IllegalArgumentException();
		if(refsWithIDs == null) refsWithIDs = Collections.<Integer, Collection<? extends ISegment>>emptyMap();
		if(refsNoIDs == null) refsNoIDs = Collections.<Collection<? extends ISegment>>emptySet();
		if(refsWithIDs.isEmpty() && refsNoIDs.isEmpty()) {
			throw new IllegalArgumentException();
		}
				
		// create references that already have an id
		int id = -1;
		List<IReference> refs = new ArrayList<IReference>(refsWithIDs.size());
		for(Entry<Integer, ? extends Collection<? extends ISegment>> entry : refsWithIDs.entrySet()){
			refs.add(new Reference(entry.getKey(), entry.getValue()));
			id = Math.max(id, entry.getKey());
		}
		
		// create references without ids
		id++;
		for(Collection<? extends ISegment> segments : refsNoIDs){
			refs.add(new Reference(id++, segments));
		}
		
		// assign final fields
		this.name = name.trim();
		this.refs = new ImmutableHasPositionSet<IReference>(refs);
	}
	/* 
	 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReferent#getName()
	 */
	public String getName(){
		return name;
	}
	
	public IHasPositionSet<IReference> getReferences(){
		return refs;
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReferent#getReference(int)
	 */
	
	public IReference getReference(int id) {
		for(IReference ref : refs) if(ref.getID() == id) return ref;
		return null;
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
	 */
	
	public int getLength() {
		return refs.getLength();
	}
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
	 */
	
	public int getOffset() {
		return refs.getOffset();
	}
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
	 */
	
	public int getRightOffset() {
		return refs.getRightOffset();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#getDisplayPosition()
	 */
	
	public IHasPosition getDisplayPosition() {
		return refs.first();
	}
	
	/* (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#calculatePosition() */
	public IHasPosition calculatePosition() {
		return PositionUtils.makePosition(refs);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IStructuredData#recalculate(edu.mit.story.core.model.IStoryModel)
	 */
	public Referent recalculate(IDesc container, IStoryModel model) {
		
		boolean unchanged = true;
		ISegment newSeg;
		
		// look for changed segments
		outermost:for(IReference ref : refs){
			for(ISegment oldSeg : ref.getSegments()){
				newSeg = (ISegment)oldSeg.recalculate(container, model);
				if(newSeg != oldSeg) {
					unchanged = false;
					break outermost;
				}
			}
		}
		
		// if no segments changed, return same
		if(unchanged) return this;
		
		// otherwise, recalculate
		Map<Integer, Set<ISegment>> newRefs = new HashMap<Integer, Set<ISegment>>(refs.size());
		Set<ISegment> newSegs;
		for(IReference oldRef : refs){
			newSegs = new HashSet<ISegment>(oldRef.getSegments().size());
			for(ISegment oldSeg : oldRef.getSegments()){
				newSeg = (ISegment)oldSeg.recalculate(container, model);
				if(newSeg != null) newSegs.add(newSeg);
			}
			if(!newSegs.isEmpty()) newRefs.put(oldRef.getID(), newSegs);
		}
		
		return newRefs.isEmpty() ? null : new Referent(name, newRefs, null);
	}

	/* 
	 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
	 */
	public boolean equals(IData tgtData, IAlignedStoryModel model) {
		if(this == tgtData)
			return true;
		if(!IReferent.class.isAssignableFrom(tgtData.getClass()))
			return false;
		
		IReferent tgtRef = (IReferent)tgtData;
		return Data.equals(refs, tgtRef.getReferences(), model);
	}
	/* 
	 * (non-Javadoc) @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(64);
		
		sb.append("ref[");
		sb.append(getName());
		sb.append('@');
		sb.append("[off=");
		sb.append(Integer.toString(getOffset()));
		sb.append(",len=");
		sb.append(Integer.toString(getLength()));
		sb.append("]: ");
		for(Iterator<IReference> i = refs.iterator(); i.hasNext(); ){
			sb.append(i.next().toString());
			if(i.hasNext()) sb.append(", ");
		}
		sb.append(']');
		return sb.toString();
	}
	
	public static IDesc findParent(IReference ref, IStoryData data){
		IReferent referent = ref.getParent();
		if(!data.supports(ReferentRep.getInstance()))
			return null;
		for(IDesc d : data.getDescriptions(ReferentRep.getInstance())){
			if(d.getData() != referent) 
				continue;
			return d;
		}
		return null;
	}
	public static IReference isReferenceInModel(IHasPosition p, IStoryData data){
		
		// if model doesn't support referents, can't be a reference
		if(!data.supports(ReferentRep.getInstance())) 
			return null;
		
		// if position is not a valid segment, can't be a reference 
		if(!Segment.isValidSegmentPosition(p, TokenRep.getInstance(), data)) 
			return null;
		
		// make the segment and check it
		ISegment segment = new Segment(p, TokenRep.getInstance(), data);
		return isReferenceInModel(segment, data);
	}

	/**
	 * Returns whether the specified description is contained as a reference in
	 * any entity description in the specified data provider. If the specified
	 * provider data doesn't support entities, the method returns {@code null}.
	 * 
	 * @throws NullPointerException
	 *             if either argument is null
	 */
	public static IReference isReferenceInModel(ISegment segment, IStoryData data){
		
		// if model doesn't support referents, can't be a reference
		if(!data.supports(ReferentRep.getInstance())) 
			return null;
		
		// search all referents for this reference
		IDescSet refs = data.getDescriptions(ReferentRep.getInstance());
		IReference r;
		for(IDesc ref : refs){ 
			r = isReferenceForReferent(segment, ref);
			if(r != null)
				return r;
		}
		return null;
	}

	/**
	 * Returns whether the specified segment is a reference of the specified
	 * entity description.
	 * 
	 * @throws NullPointerException
	 *             if either argument is null
	 * @throws ClassCastException
	 *             if the entity description is not of the correct type
	 */
	public static IReference isReferenceForReferent(ISegment segment, IDesc desc){
		Referent data = (Referent)desc.getData();
		for(IReference ref : data.getReferences()){
			for(ISegment seg : ref.getSegments()){
				if(PositionUtils.equalAsPositions(seg, segment)) 
					return ref;
			}
		}
		return null;
	}

	public static String serialize(IReferent ref){
		
		List<String> blocks = new ArrayList<String>(ref.getReferences().size()+1);
		
		// name is first field
		blocks.add(ref.getName());
		
		// all other fields are references
		for(IReference r : ref.getReferences()) blocks.add(serialize(r));
		
		// everything is separated by pipes
		return BlockMapper.PIPE.serialize(blocks);
		
	}
	
	public static String serialize(IReference ref){
		StringBuilder sb = new StringBuilder(64);

		// reference sub-fields are colon delimited
		// first reference sub-field is the reference id
		sb.append(Integer.toString(ref.getID()));
		sb.append(':');
		
		// second reference sub-field contains the reference's segments
		sb.append(SegmentMapper.serializeStatic(ref.getSegments()));
		return sb.toString();
	}
	
	public static Referent reconstitute(String input, IStoryData data){
		List<String> blocks = BlockMapper.PIPE.reconstitute(input);
		
		// create the reference map
		Map<Integer, Collection<? extends ISegment>> refs = new HashMap<Integer, Collection<? extends ISegment>>(blocks.size()-1);
		IValueMapper<ISegment> mapper = new SegmentMapper(TokenRep.getInstance(), data);
		
		// loop variables
		String block;
		int idx;
		int id;
		List<ISegment> segments;
		
		// convert each block after the first into an id and set of segments
		for(int i = 1; i < blocks.size(); i++){
			block = blocks.get(i).trim();
			
			// find the colon delimiter
			idx = block.indexOf(':');
			
			// extract the id and segments
			id = Integer.parseInt(block.substring(0, idx));
			segments = mapper.reconstitute(block.substring(idx+1));
			
			refs.put(id, segments);
		}
		
		return new Referent(blocks.get(0), refs, null);
		
	}
	

	
	protected class Reference implements IReference {
		
		// instance fields set on construction
		private final int id;
		private final ISegmentSet segments;
		private transient String text;
		
		public Reference(int id, Collection<? extends ISegment> segments){
			if(segments.isEmpty()) throw new IllegalArgumentException();
			if(id < 0) throw new IllegalArgumentException();
			this.id = id;
			this.segments = new SegmentSet(segments);
		}

		/* 
		 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReference#getID()
		 */
		
		public int getID() {
			return id;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReference#getReferent()
		 */
		
		public IReferent getParent() {
			return Referent.this;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.discourse.core.rep.referent.IReference#getSegments()
		 */
		
		public ISegmentSet getSegments() {
			return segments;
		}
		
		public ISegmentSet getExpression() {
			return segments;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getLength()
		 */
		
		public int getLength() {
			return segments.getLength();
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getOffset()
		 */
		
		public int getOffset() {
			return segments.getOffset();
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.position.IHasPosition#getRightOffset()
		 */
		
		public int getRightOffset() {
			return segments.getRightOffset();
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.util.IHasDisplayText#getDisplayText()
		 */
		
		public String getDisplayText() {
			if(text == null) text = Segment.generateText(segments);
			return text;
		}

		/* 
		 * (non-Javadoc) @see java.lang.Object#toString()
		 */
		
		public String toString() {
			String posStr = "[off=" + getOffset() + ",len=" + getLength() + "](" + getID() + ")";
			return "refto[" + getName() + " - " + Segment.generateText(getSegments()) + ']' + posStr;
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.rep.IHasRep#getRep()
		 */
		public IRep getRep() {
			return segments.first().getRep();
		}

		/* 
		 * (non-Javadoc) @see edu.mit.story.core.desc.IData#equals(edu.mit.story.core.desc.IData, edu.mit.story.core.align.IAlignedStoryModel)
		 */
		public boolean equals(IData tgtData, IAlignedStoryModel model) {
			if(this == tgtData)
				return true;
			if(!IReference.class.isAssignableFrom(tgtData.getClass()))
				return false;
			
			IReference tgtRef = (IReference)tgtData;
			return segments.equals(tgtRef.getExpression(), model);
		}

	}

}
