/* Filename: IndicationStructureController.java
 * Creator: Raquel Hervas
 * Format: Java 2 v1.6.0
 * Date created: 30/09/2009
 */
package nil.ucm.indications2.ui.controllers;

import nil.ucm.indications2.core.rep.IModifier;
import nil.ucm.indications2.core.rep.INucleus;
import nil.ucm.indications2.core.rep.IndicationStructureRep;
import nil.ucm.indications2.core.rep.IndicationType;
import nil.ucm.indications2.core.rep.model.IIndicationStructureModel;
import nil.ucm.indications2.core.rep.model.IModifierModel;
import nil.ucm.indications2.core.rep.model.INucleusModel;
import nil.ucm.indications2.core.rep.model.IndicationStructureModel;
import nil.ucm.indications2.core.rep.model.ModifierModel;
import nil.ucm.indications2.core.rep.model.NucleusModel;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider.ITypedList;
import nil.ucm.indications2.ui.content.IndicationStructureContentProvider.ReferentReferencePair;

import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;

import edu.mit.story.core.desc.IDesc;
import edu.mit.story.core.desc.ISegment;
import edu.mit.story.core.desc.IStructuredData;
import edu.mit.story.core.meta.timing.ITiming;
import edu.mit.story.core.meta.timing.Timing;
import edu.mit.story.core.model.change.ChangeUtils;
import edu.mit.story.core.position.IHasPosition;
import edu.mit.story.core.position.IHasPositionSet;
import edu.mit.story.ui.actions.IUpdatingAction;
import edu.mit.story.ui.actions.SelectionUtils;
import edu.mit.story.ui.actions.generic.DeleteAction;
import edu.mit.story.ui.actions.generic.EditAction;
import edu.mit.story.ui.editor.IStoryEditor;

/** 
 *   Specific controller for the indication structures.  
 * Used in the Indication Structure creator.
 *
 * @author Raquel Hervas
 * @version 1.0, (Jan. 11, 2010)
 * @since nil.ucm.indications.ui 1.0.0
 */
public class IndicationStructureController implements IIndicationStructureController {
	
	//Models
	private final IStoryEditor editor;
	private final IIndicationStructureModel model;
	
	//Controllers
	private final IReferenceController refController;
	private final INucleusController nucController;
	private final IModifierController modController;
	
	private DeleteElementAction deleteAction;
	private EditElementAction editAction;
	
	public IndicationStructureController(IStoryEditor ed, String refAnnoID, String nucAnnoID, String modAnnoID) {
		this.editor = ed;
		this.model = new IndicationStructureModel(editor.getStoryModel());
		this.refController = new ReferenceController(model.getReference(), refAnnoID, editor);
		this.nucController = new NucleusController(editor, nucAnnoID);
		this.modController = new ModifierController(editor, modAnnoID);
	}
	
	/* 
	 * (non-Javadoc) @see nil.ucm.indications.ui.controllers.IIndicationStructureController#getStoryEditor()
	 */
	
	public IStoryEditor getStoryEditor() {
		return editor;
	}

	
	public IIndicationStructureModel getIndicationStructureModel() {
		return model;
	}
		
	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.controllers.IReferentController#getNewRefController()
	 */
	public IReferenceController getReferenceController() {
		return refController;
	}

	public INucleusController getNucleusController() {
		return nucController;
	}
	
	
	public IModifierController getModifierController() {
		return modController;
	}
	
	public String getName() {
		return model.getName();
	}
	
//	public void addReference() {
//		
//		// get segments
////		IIndArgModel refModel = indArgController.getArgModel();
//		IReferenceModel refModel = refController.getModel();
//		IHasPositionSet<ISegment> segs = refModel.getSegments();
////		IndicationType type = refModel.getIndicationType();
//		if(segs.isEmpty()) return;
//
////		// obtain appropriate reference model
//		IReference loaded = refModel.getLoaded();
////		IIndArgModel ref;
//		IReferenceModel ref;
//		if(loaded == null){
//			// if we are not manipulating a current reference, load the corresponding one
//			// Remember that the reference already exists!!
//			// We need the referent associated with the reference
//			IHasPosition selectedPos = editor.getSelectedRange();
//			IDescSet possibleReferents = editor.getStoryModel().getData().getDescriptions(ReferentRep.getInstance()).openSet(selectedPos);
//			IDesc firstPossibleReferent = null;
//			
//			//We can find three different situations:
//			// - There is only one referent. Then it is the one we use.
//			// - There is no referent. Then we create if with the corresponding reference
//			// - There is more than one referent. We present the user the choices.
//			
//			//There is no referent. We create one.
//			if (possibleReferents.size() == 0){
//				createNewReferent(segs);
//				
//				//We calculate again the possible referents, but we have to wait until it has been added (BOTCH, but the lock of the change is not working!!)
//				while (possibleReferents.size() == 0) {
//					possibleReferents = editor.getStoryModel().getData().getDescriptions(ReferentRep.getInstance()).openSet(selectedPos);
//				}
//				
//				firstPossibleReferent = possibleReferents.first();
//			}
//			//There is more than one possible referent. We present the options to the user
//			else if (possibleReferents.size() > 1) {
//
//				while (firstPossibleReferent == null) {
//					
//					firstPossibleReferent = ReferentSelectionDialog.open(editor, null);
//					
//					if (!possibleReferents.contains(firstPossibleReferent)) {
//						
//						firstPossibleReferent = null;
//					}
//					  
//				}
//			}
//			else {
//				firstPossibleReferent = possibleReferents.first();
//			}
//			
//			//At this point we always have one and just one referent
//			model.addReferent(firstPossibleReferent);
//			
//			IDesc referentDesc = firstPossibleReferent;
//			IReferent referent = (IReferent)referentDesc.getData();
//			IHasPositionSet<IReference> setReferences = referent.getReferences();
//			IHasPositionSet<IReference> possibleReferences = setReferences.openSet(selectedPos);
//			
//			//There is always a reference in the same position of the referent
//			//There must no be more than one reference in the same position for the same referent. We use the first one.
//			IReference reference = possibleReferences.first();
//			model.addReference(reference);
//			
////		} else if(loaded instanceof IIndArgModel){
//		} else if(loaded instanceof IReferenceModel){
//			// if we are manipulating an existing reference model, update it
//			ref = (IReferenceModel)loaded;
//			ref.setSegments(segs);
////			ref.setIndicationType(type);
//		} 
//		
//		refModel.clear();
//		refController.clear();
//	}
//
//	/** 
//	 *   Creates a referent with a reference when a piece of text is selected for creating an indication but 
//	 * it does not contain a referent.
//	 */
//	private void createNewReferent(IHasPositionSet<ISegment> segs) {
//		IReferentModel newReferentModel = new ReferentModel(editor.getStoryModel());
//		IReferenceModel newReferenceModel = new ReferenceModel(editor.getStoryModel());
//		newReferenceModel.setSegments(segs);
//		newReferentModel.addReference(newReferenceModel);
//		
//		IStructuredData data = newReferentModel.toReferent();
//		long id = editor.getStoryModel().getNextID();
//		IHasPosition pos = data.calculatePosition();
//		// FIXME: put in timing data here, see ticket#41
//		IDesc refDesc = new Desc(id, ReferentRep.getInstance(),pos, data, null, null, true);
//		
//		List<IDesc> toAdd = new LinkedList<IDesc>();
//		toAdd.add(refDesc);
//		IModelChange change = new AddDescriptionsChange(toAdd);
//		editor.getStoryModel().applyChange(this, change, false);
//	}

	
	public void addNucleus() {
		
		// get segments
		INucleusModel nucModel = nucController.getModel();
		IHasPositionSet<ISegment> segs = nucModel.getSegments();
//		IndicationType type = nucModel.getIndicationType();
		IndicationType type = nucModel.getType();
		if(segs.isEmpty()) return;

		// obtain appropriate reference model
		Object loaded = nucModel.getLoaded();
		INucleusModel nuc;
		if(loaded == null){
			// if we are not manipulating a current nucleus, create anew
			nuc = new NucleusModel(editor.getStoryModel());
			nuc.setSegments(segs);
//			nuc.setIndicationType(type);
			nuc.setType(type);
			model.addNucleus(nuc);
			nucModel.clear();
		} else if(loaded instanceof INucleusModel){
			// if we are manipulating an existing reference model, update it
			nuc = (INucleusModel)loaded;
			nuc.setSegments(segs);
//			nuc.setIndicationType(type);
			nuc.setType(type);
			nucModel.clear();
		} 
	}
	
	
	public void addModifier() {
		
		// get segments
		IModifierModel modModel = modController.getModel();
		IHasPositionSet<ISegment> segs = modModel.getSegments();
		IndicationType type = modModel.getType();
		if(segs.isEmpty()) return;

		// obtain appropriate reference model
		Object loaded = modModel.getLoaded();
		IModifierModel mod;
		if(loaded == null){
			// if we are not manipulating a current modifier, create anew
			mod = new ModifierModel(editor.getStoryModel());
			mod.setSegments(segs);
			mod.setType(type);
			model.addModifier(mod);
			modModel.clear();
		} else if(loaded instanceof IModifierModel){
			// if we are manipulating an existing reference model, update it
			mod = (IModifierModel)loaded;
			mod.setSegments(segs);
			mod.setType(type);
			modModel.clear();
		} 
	}
	
	public void removeModifier(IModifierModel mod) {
		model.removeModifier(mod);
	}
	
	public void removeNucleus(INucleusModel nuc) {
		model.removeNucleus(nuc);
	}

	/* 
	 * (non-Javadoc) @see nil.ucm.indications2.ui.controllers.IIndicationStructureController#toggleCopularPredicate()
	 */
	public void toggleCopularPredicate() {
		model.setCopularPredicate(!model.isCopularPredicate());
	}

	public void commit() {
		
		if(model.getMessageType() > IMessageProvider.INFORMATION) return;
			
		ITiming timing = new Timing(model.getStart(), ITiming.SOURCE_USER);
		// make indication structure description
		IStructuredData data = model.toIndicationStructure();
		IHasPosition p = data.calculatePosition();
		ChangeUtils.edit(this, editor.getStoryModel(), IndicationStructureRep.getInstance(), p, data, timing, model.getLoaded());

		clear();
	}


	
	public void dispose() {
		this.nucController.dispose();
		this.modController.dispose();
		this.refController.dispose();
	}
	
	/* 
	 * (non-Javadoc) @see edu.mit.discourse.ui.controllers.IReferentController#clear()
	 */
	
	public void clear() {
		model.clear();
		nucController.clear();
		modController.clear();
		refController.clear();
	}
	

	
	public void load(IDesc desc) {
		clear();
		model.load(desc);
	}
	
	public DeleteElementAction getActionDelete(){
		if(deleteAction == null) deleteAction = new DeleteElementAction();
		return deleteAction;
	}
	
	public EditElementAction getActionEdit(){
		if(editAction == null) editAction = new EditElementAction();
		return editAction;
	}

	/** 
	 *   Deletion of elements in the indication structure
	 *
	 * @author Raquel Hervas
	 * @version 1.0, (Jan. 11, 2010)
	 */
	protected class DeleteElementAction extends DeleteAction implements IUpdatingAction {
		
		INucleusModel nuc;
		IModifierModel mod;		
		
		/* 
		 * (non-Javadoc) @see edu.mit.story.ui.actions.generic.DeleteAction#doDelete()
		 */
		
		protected void doDelete() {
			//The element to be deleted will be in its corresponding variable
			if (mod != null) {
				removeModifier(mod);
			}
			
			if(nuc != null) {
				removeNucleus(nuc);
			}
		}

		/* 
		 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		
		public void selectionChanged(SelectionChangedEvent event) {
			//We have selected a Indication Structure model
			IIndicationStructureModel indStructModel = SelectionUtils.extractSingle(event.getSelection(), IIndicationStructureModel.class);
			if (indStructModel != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected the ReferentReferencePair 
			ReferentReferencePair pair = SelectionUtils.extractSingle(event.getSelection(), ReferentReferencePair.class);
			if (pair != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a list of nucleus or modifiers
			ITypedList list = SelectionUtils.extractSingle(event.getSelection(), ITypedList.class);
			if (list != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a nucleus
			nuc = SelectionUtils.extractSingle(event.getSelection(), INucleusModel.class);
			if (nuc != null){
				setEnabled(true);
				return;				
			}
			
			//We have selected a modifier
			mod = SelectionUtils.extractSingle(event.getSelection(), IModifierModel.class);
			if (mod != null) {
				setEnabled(true);
				return;
			}
		}
		
	}
	
	/** 
	 *   Action for editing the elements of the indication structure
	 *
	 * @author Raquel
	 * @version 1.0, (Oct. 21, 2009)
	 */
	protected class EditElementAction extends EditAction implements IUpdatingAction {
		
		INucleus nuc;
		IModifier mod;	

		/* 
		 * (non-Javadoc) @see edu.mit.story.ui.actions.generic.EditAction#doEdit()
		 */
		
		protected void doEdit() {
			if (nuc != null) getNucleusController().load(nuc);
			if (mod != null) getModifierController().load(mod);
		}

		/* 
		 * (non-Javadoc) @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		
		public void selectionChanged(SelectionChangedEvent event) {
			//We have selected a Indication Structure model
			IIndicationStructureModel indStructModel = SelectionUtils.extractSingle(event.getSelection(), IIndicationStructureModel.class);
			if (indStructModel != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected the ReferentReferencePair 
			ReferentReferencePair pair = SelectionUtils.extractSingle(event.getSelection(), ReferentReferencePair.class);
			if (pair != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a list of nucleus or modifiers
			ITypedList list = SelectionUtils.extractSingle(event.getSelection(), ITypedList.class);
			if (list != null) {
				setEnabled(false);
				return;
			}
			
			//We have selected a nucleus
			nuc = SelectionUtils.extractSingle(event.getSelection(), INucleus.class);
			if (nuc != null){
				setEnabled(true);
				return;				
			}
			
			//We have selected a modifier
			mod = SelectionUtils.extractSingle(event.getSelection(), IModifier.class);
			if (mod != null) {
				setEnabled(true);
				return;
			}
		}
		
	}

}
